package com.example.matteribc

import androidx.appcompat.app.AppCompatActivity
import android.os.Bundle
import android.Manifest
import android.annotation.SuppressLint
import android.content.ActivityNotFoundException
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.net.Uri
import android.util.Log
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.core.content.ContextCompat
import java.util.concurrent.Executors
import androidx.camera.core.*
import androidx.camera.lifecycle.ProcessCameraProvider
import androidx.fragment.app.DialogFragment
import com.google.common.util.concurrent.ListenableFuture
import com.google.mlkit.vision.barcode.Barcode
import com.google.mlkit.vision.barcode.BarcodeScanner
import com.google.mlkit.vision.barcode.BarcodeScannerOptions
import com.google.mlkit.vision.barcode.BarcodeScanning
import com.google.mlkit.vision.common.InputImage
import kotlinx.android.synthetic.main.activity_main.*
import java.util.concurrent.ExecutorService

/**
 * Main Activity of the Intent-Based Commissioning sample application.
 */
class MainActivity : AppCompatActivity(),
  ConfirmCommissioningDialogFragment.ConfirmCommissioningDialogListener {

  private var imageCapture: ImageCapture? = null
  private var cameraProvider: ProcessCameraProvider? = null

  private lateinit var cameraProviderFuture: ListenableFuture<ProcessCameraProvider>
  private lateinit var permissionsLauncher: ActivityResultLauncher<Array<String>>
  private lateinit var cameraExecutor: ExecutorService

  // ---------------------------------------------------------------------------
  // Lifecycle Events
  // TODO: remove unused lifecycle methods and logs from lifecycle events after
  //       first CL of the app is approved.

  override fun onCreate(savedInstanceState: Bundle?) {
    super.onCreate(savedInstanceState)
    Log.d(TAG, "lifecycle [onCreate]")
    setContentView(R.layout.activity_main)

    cameraProviderFuture = ProcessCameraProvider.getInstance(this)

    permissionsLauncher =
      registerForActivityResult(ActivityResultContracts.RequestMultiplePermissions()) {
        if (allPermissionsGranted()) {
          startCamera()
        } else {
          finish()
        }
      }

    cameraExecutor = Executors.newSingleThreadExecutor()
  }

  override fun onStart() {
    super.onStart()
    Log.d(TAG, "lifecycle [onStart]")
  }

  override fun onRestart() {
    super.onRestart()
    Log.d(TAG, "lifecycle [onRestart]")
  }

  override fun onResume() {
    super.onResume()
    Log.d(TAG, "lifecycle [onResume]")
    if (allPermissionsGranted()) {
      Log.d(TAG, "lifecycle [onResume] before startCamera()")
      startCamera()
      Log.d(TAG, "lifecycle [onResume] after startCamera()")
    } else {
      Log.d(TAG, "lifecycle [onResume] before permissionsLauncher.launch()")
      permissionsLauncher.launch(REQUIRED_PERMISSIONS)
      Log.d(TAG, "lifecycle [onResume] after permissionsLauncher.launch()")
    }
  }

  override fun onPause() {
    super.onPause()
    Log.d(TAG, "lifecycle [onPause]")
    cameraProvider?.unbindAll()
  }

  override fun onStop() {
    super.onStop()
    Log.d(TAG, "lifecycle [onStop]")
  }

  override fun onDestroy() {
    super.onDestroy()
    Log.d(TAG, "lifecycle [onDestroy]")
    cameraExecutor.shutdown()
  }

  // ---------------------------------------------------------------------------
  // Permissions

  private fun allPermissionsGranted() = REQUIRED_PERMISSIONS.all {
    ContextCompat.checkSelfPermission(baseContext, it) == PackageManager.PERMISSION_GRANTED
  }

  // ---------------------------------------------------------------------------
  // Camera processing

  private fun startCamera() {
    cameraProviderFuture.addListener({
      // Used to bind the lifecycle of cameras to the lifecycle owner
      val cameraProvider = cameraProviderFuture.get()
      this.cameraProvider = cameraProvider

      // Preview
      val preview = Preview.Builder()
        .build()
        .also {
          it.setSurfaceProvider(viewFinder.surfaceProvider)
        }

      // ImageCapture
      imageCapture = ImageCapture.Builder()
        .build()

      // ImageAnalyzer against Barcode
      val options = BarcodeScannerOptions.Builder()
           .setBarcodeFormats(Barcode.FORMAT_QR_CODE)
           .build()
      val barcodeScanner: BarcodeScanner = BarcodeScanning.getClient(options)
      val imageAnalyzer = ImageAnalysis.Builder()
        .build()
        .also {
          it.setAnalyzer(cameraExecutor, { imageProxy ->
            processImageProxy(barcodeScanner, imageProxy)
          })
        }

      // Select back camera as a default
      val cameraSelector = CameraSelector.DEFAULT_BACK_CAMERA

      try {
        // Unbind use cases before rebinding
        cameraProvider.unbindAll()
        // Bind use cases to camera
        cameraProvider.bindToLifecycle(
          this, cameraSelector, preview, imageCapture, imageAnalyzer
        )
      } catch (e: Exception) {
        Log.e(TAG, "Use case binding failed", e)
      }
    }, ContextCompat.getMainExecutor(this))
  }

  @SuppressLint("UnsafeOptInUsageError")
  private fun processImageProxy(
    barcodeScanner: BarcodeScanner,
    imageProxy: ImageProxy
  ) {
    val inputImage =
      InputImage.fromMediaImage(imageProxy.image!!, imageProxy.imageInfo.rotationDegrees)

    barcodeScanner.process(inputImage)
      .addOnSuccessListener { barcodes ->
        barcodes.forEach {
          Log.d(TAG, "qrCode: [${it.rawValue!!}]")
          val isMatterQrCode = isMatterQrCode(it.rawValue!!)
          Log.d(TAG, "*** isMatterQrCode [$isMatterQrCode]")
          if (isMatterQrCode) {
            // stop processing right away
            cameraProvider!!.unbindAll()

            val dialog = ConfirmCommissioningDialogFragment.newInstance(it.rawValue!!)
            dialog.show(supportFragmentManager, null)
          }
        }
      }
      .addOnFailureListener {
        Log.e(TAG, "Failed to process camera image", it)
      }.addOnCompleteListener {
        // When the image is from CameraX analysis use case, must call image.close() on received
        // images when finished using them. Otherwise, new images may not be received or the camera
        // may stall.
        imageProxy.close()
      }
  }

  // The dialog fragment receives a reference to this Activity through the
  // Fragment.onAttach() callback, which it uses to call the following methods
  // defined by the ConfirmCommissioningDialogFragment.ConfirmCommissioningDialogListener interface
  override fun onDialogPositiveClick(dialog: DialogFragment, qrCode: String) {
    startCommissioning(this, qrCode)
  }

  override fun onDialogNegativeClick(dialog: DialogFragment) {
    // User touched the dialog's negative button
    startCamera()
  }

  // ---------------------------------------------------------------------------
  // Matter functions

  private fun isMatterQrCode(value: String): Boolean {
    return value.matches(Regex("""MT:[A-Z0-9.-]{19,}"""))
  }

  private fun startCommissioning(context: Context, qrCodeString: String): Boolean {
    Log.d(TAG, "startCommissioning() [$qrCodeString]")
    val intent =
      Intent(Intent.ACTION_VIEW)
        .setData(Uri.parse(qrCodeString))
        .setPackage("com.google.android.gms")
    return try {
      context.startActivity(intent)
      Log.d(TAG, "startCommissioning() Success!")
      true
    } catch (ex: ActivityNotFoundException) {
      // Supporting Play Services version not available.
      Log.d(TAG, "startCommissioning() ActivityNotFoundException")
      false
    }
  }

  // ---------------------------------------------------------------------------
  // Companion object

  companion object {
    private const val TAG = "IBC_MainActivity"
    private val REQUIRED_PERMISSIONS = arrayOf(Manifest.permission.CAMERA)
  }
}